//	TorusGamesGameChoiceController.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesGameChoiceController.h"
#import "GeometryGamesUtilities-iOS.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesLocalization.h"


static const struct
{
	GameType	itsGame;
	Char16		*itsName,
				*itsImageName;
}
gGame2DNames[] =
{
	{Game2DTicTacToe,	u"Tic-Tac-Toe",		u"Table Images/Game/2D TicTacToe"	},
	{Game2DGomoku,		u"Gomoku",			u"Table Images/Game/2D Gomoku"		},
	{Game2DMaze,		u"Maze",			u"Table Images/Game/2D Maze"		},
	{Game2DCrossword,	u"Crossword",		u"Table Images/Game/2D Crossword"	},
	{Game2DWordSearch,	u"Word Search",		u"Table Images/Game/2D Word Search"	},
	{Game2DJigsaw,		u"Jigsaw Puzzle",	u"Table Images/Game/2D Jigsaw"		},
	{Game2DChess,		u"Chess",			u"Table Images/Game/2D Chess"		},
	{Game2DPool,		u"Pool",			u"Table Images/Game/2D Pool"		},
	{Game2DApples,		u"Apples",			u"Table Images/Game/2D Apples"		}
},
gGame3DNames[] =
{
	{Game3DTicTacToe,	u"Tic-Tac-Toe",		u"Table Images/Game/3D TicTacToe"	},
	{Game3DMaze,		u"Maze",			u"Table Images/Game/3D Maze"		}
},
*gGameNames[2] =
{
	gGame2DNames,
	gGame3DNames
};


@implementation TorusGamesGameChoiceController
{
	id<TorusGamesGameChoiceDelegate> __weak	itsDelegate;
	GameType								itsGameChoice;
	UIBarButtonItem							*itsCancelButton;
}


- (id)initWithDelegate:(id<TorusGamesGameChoiceDelegate>)aDelegate gameChoice:(GameType)aGameChoice
{
	self = [super initWithStyle:UITableViewStyleGrouped];
	if (self != nil)
	{
		itsDelegate		= aDelegate;
		itsGameChoice	= aGameChoice;
	
		//	To make it easy to try Crossword and Word Search puzzles
		//	in different languages, Torus Games lets the player change
		//	languages on the fly.  So let's provide an explicit localization
		//	for the "Cancel" button, in lieu of the built-in
		//
		//		initWithBarButtonSystemItem:UIBarButtonSystemItemCancel
		//
		itsCancelButton = [[UIBarButtonItem alloc]
			initWithTitle:	GetLocalizedTextAsNSString(u"Cancel")
			style:			UIBarButtonItemStylePlain
			target:			aDelegate
			action:			@selector(userDidCancelGameChoice)];

		if (aGameChoice == GameNone)
		{
			//	At launch, when no game is yet present,
			//	use the longer, more informative title "Choose a Game".
			//	To encourage the user to choose a game,
			//	adaptNavBarForHorizontalSize will suppress the Cancel button,
			//	so there'll be plenty of room for the longer title.
			[[self navigationItem] setTitle:GetLocalizedTextAsNSString(u"Choose a Game")];
		}
		else	//	The user is changing from one game to another.
		{
			//	Use the short title "Game", to allow space for the Cancel button
			//	and also because the short title sounds better for repeated use.
			[[self navigationItem] setTitle:GetLocalizedTextAsNSString(u"Game")];
		}
	}
	return self;
}

- (BOOL)prefersStatusBarHidden
{
	return [itsDelegate prefersStatusBarHidden];
}

- (void)viewWillAppear:(BOOL)animated
{
	[super viewWillAppear:animated];
	
	//	We must call -layoutIfNeeded explicitly, otherwise -contentSize returns CGSizeZero.
	[[self tableView] layoutIfNeeded];
	[self setPreferredContentSize:(CGSize){	PREFERRED_POPOVER_WIDTH,
											[[self tableView] contentSize].height}];

	[self adaptNavBarForHorizontalSize:
		[[RootViewController(self) traitCollection] horizontalSizeClass]];
}


#pragma mark -
#pragma mark GeometryGamesPopover

- (void)adaptNavBarForHorizontalSize:(UIUserInterfaceSizeClass)aHorizontalSizeClass
{
	if (itsGameChoice == GameNone)
	{
		//	At launch, when no game is yet present,
		//	suppress the Cancel button to encourage
		//	the user to choose a game.  Indeed,
		//	with compact width this forces the user
		//	to choose a game.
		[[self navigationItem] setRightBarButtonItem:nil animated:NO];
	}
	else
	{
		[[self navigationItem]
			setRightBarButtonItem:	(aHorizontalSizeClass == UIUserInterfaceSizeClassCompact ?
										itsCancelButton : nil)
			animated:				NO];
	}
}


#pragma mark -
#pragma mark UITableViewDataSource

- (NSInteger)numberOfSectionsInTableView:(UITableView *)aTableView
{
	return 2;
}

- (NSString *)tableView:(UITableView *)aTableView titleForHeaderInSection:(NSInteger)aSection
{
	switch (aSection)
	{
		case 0:		return GetLocalizedTextAsNSString(u"2D Games");
		case 1:		return GetLocalizedTextAsNSString(u"3D Games");
		default:	return nil;
	}
}

- (NSInteger)tableView:(UITableView *)aTableView numberOfRowsInSection:(NSInteger)aSection
{
	switch (aSection)
	{
		case 0:		return BUFFER_LENGTH(gGame2DNames);
		case 1:		return BUFFER_LENGTH(gGame3DNames);
		default:	return 0;
	}
}

- (UITableViewCell *)tableView:(UITableView *)aTableView cellForRowAtIndexPath:(NSIndexPath *)anIndexPath
{
	UITableViewCell	*theCell;
	UILabel			*theLabel;
	NSUInteger		theSection,
					theRow;
	NSString		*theTitle;
	bool			theCheckmark;

	theCell			= [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:nil];
	theLabel		= [theCell textLabel];
	theSection		= [anIndexPath section];
	theRow			= [anIndexPath row];
	
	theTitle		= GetLocalizedTextAsNSString(gGameNames[theSection][theRow].itsName);
	if (IsCurrentLanguage(u"vi"))
	{
		//	In Vietnamese, "Cờ ca-rô 9 ô" might be inappropriate
		//	for a 3×3×3 game with 27 cells.  So use the full 3D names
		//	"Cờ ca-rô 3D" and "Mê cung 3D" instead.
		switch (gGameNames[theSection][theRow].itsGame)
		{
			case Game3DTicTacToe:
				theTitle = GetLocalizedTextAsNSString(u"3D Tic-Tac-Toe");
				break;
			
			case Game3DMaze:
				theTitle = GetLocalizedTextAsNSString(u"3D Maze");
				break;
			
			default:
				//	Keep the standard title as assigned above.
				break;
		}
	}
	theCheckmark	= (gGameNames[theSection][theRow].itsGame == itsGameChoice);

	[theLabel setText:theTitle];
	[theCell setAccessoryType:(theCheckmark ? UITableViewCellAccessoryCheckmark : UITableViewCellAccessoryNone)];
	[[theCell imageView] setImage:[UIImage imageNamed:
		GetNSStringFromZeroTerminatedString(gGameNames[theSection][theRow].itsImageName)]];

	return theCell;
}


#pragma mark -
#pragma mark UITableViewDelegate

- (NSIndexPath *)tableView:(UITableView *)aTableView willSelectRowAtIndexPath:(NSIndexPath *)anIndexPath
{
	NSUInteger	theSection,
				theRow;

	theSection	= [anIndexPath section];
	theRow		= [anIndexPath row];

	itsGameChoice = gGameNames[theSection][theRow].itsGame;

	//	Reload the table with the checkmark by the newly selected game,
	//	so the user will see it as the table animates away.
	[[self tableView] reloadData];

	//	Report the new game to the caller,
	//	who will dismiss this TorusGamesGameChoiceController.
	[itsDelegate userDidChooseGame:itsGameChoice];

	return nil;
}

@end
